/**
 *   RexxScriptEngine
 *   Copyright 2010 Stani Ryabenkiy
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 *   
 */
package com.sun.script.rexx;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;

import javax.script.Bindings;
import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineFactory;
import javax.script.ScriptException;
import javax.script.SimpleBindings;
import javax.script.SimpleScriptContext;

import org.apache.bsf.BSFException;
import org.apache.bsf.BSFManager;

/**
 * A JSR-223 compliant ScriptEngine for BSF 2.4 / BSF4ooRexx / ooRexx.
 * 
 * Loading the bindings each time a script is evaluated might not be the most efficient solution, but it does work. I'm not sure
 * that there is a more elegant solution given that bindings can come from diverse ScriptContext/Bindings sources - with the result
 * that the exact set of bindings to be loaded is not determined until the evaluation methods are called.
 * 
 * @author stani
 *
 */
public class RexxScriptEngine implements ScriptEngine {

	private ScriptEngineFactory factory;
	private ScriptContext context;
	
	
	public RexxScriptEngine(ScriptEngineFactory factory) {
		this.context = new SimpleScriptContext();
		this.factory = factory;

	}
	
	@Override
	public Bindings createBindings() {
		return new SimpleBindings();
	}

	@Override
	public Object eval(String script) throws ScriptException {
		return eval(script, getContext());
	}

	@Override
	public Object eval(Reader reader) throws ScriptException {
		return eval(reader, getContext());
	}

	/**
	 * declares BSF beans based on all mappings in all scopes in a given ScriptContext
	 * 
	 * @param bsfManager
	 * @param context
	 * @throws BSFException
	 */
	private void declareBeansFromScriptContext(BSFManager bsfManager, ScriptContext context) throws BSFException {
		
		// first global scope
		declareGlobalScopeBeansFromScriptContext(bsfManager, context);
		
		// then engine scope
		declareEngineScopeBeansFromScriptContext(bsfManager, context);
		
		// then any other scopes (normally none, but could optionally be implemented in a subclass of ScriptContext)		
		declareMiscScopeBeansFromScriptContext(bsfManager, context);
	}

	/**
	 * declares BSF beans based on all mappings in global scope in a given ScriptContext
	 * 
	 * @param bsfManager
	 * @param context
	 * @throws BSFException
	 */
	private void declareGlobalScopeBeansFromScriptContext(BSFManager bsfManager, ScriptContext context) throws BSFException {
		Bindings bindings = context.getBindings(ScriptContext.GLOBAL_SCOPE);
		if (bindings != null)
			declareBeansFromBindings(bsfManager, bindings);
	}
	
	/**
	 * declares BSF beans based on all mappings in engine scope in a given ScriptContext
	 * 
	 * @param bsfManager
	 * @param context
	 * @throws BSFException
	 */
	private void declareEngineScopeBeansFromScriptContext(BSFManager bsfManager, ScriptContext context) throws BSFException {
		Bindings bindings = context.getBindings(ScriptContext.ENGINE_SCOPE);
		if (bindings != null)
			declareBeansFromBindings(bsfManager, bindings);	
	}
	
	/**
	 * declares BSF beans based on all mappings in "other" scopes (ie: not global and not engine scope) in a given ScriptContext
	 * 
	 * @param bsfManager
	 * @param context
	 * @throws BSFException
	 */
	private void declareMiscScopeBeansFromScriptContext(BSFManager bsfManager, ScriptContext context) throws BSFException {
		for (Integer i : context.getScopes()) {
			if (i.intValue() != ScriptContext.ENGINE_SCOPE && i.intValue() != ScriptContext.GLOBAL_SCOPE) {
				Bindings bindings = context.getBindings(i);
				if (bindings != null)
					declareBeansFromBindings(bsfManager, bindings);
			}
			
		}	
	}
	
	/**
	 * declares BSF beans based on all mappings in a given Bindings
	 * 
	 * @param bsfManager
	 * @param bindings
	 * @throws BSFException
	 */
	private void declareBeansFromBindings(BSFManager bsfManager, Bindings bindings) throws BSFException {
		for (String key : bindings.keySet()) {
			Object value = bindings.get(key);
			if (value != null)
				bsfManager.declareBean(key, value, value.getClass());
		}
	}
	
	/**
	 * reads a String from a Reader, line by line
	 * 
	 * @param reader
	 * @return
	 * @throws IOException
	 */
	private String readStringFromReader(Reader reader) throws IOException {
		BufferedReader bufReader = new BufferedReader(reader);
		String string = "";
		
		// concatenate from reader, line by line
		String line = bufReader.readLine();
		while (line != null) {
			string += line;
			line = bufReader.readLine();
		}
		return string;
		
	}
	
	@Override
	public Object eval(String script, ScriptContext context)
			throws ScriptException {
		
		BSFManager bsfManager = new BSFManager();
		try {
			// load beans from supplied context
			declareBeansFromScriptContext(bsfManager, context);
			return bsfManager.eval("rexx", "<unknown>", 0, 0, script);
		
		}
		catch(BSFException e) {
			throw new ScriptException(e);
		}
	}

	@Override
	public Object eval(Reader reader, ScriptContext context)
			throws ScriptException {
		
		String script = "";
		try {
			script = readStringFromReader(reader);
		}
		catch (IOException e) {
			throw new ScriptException(e);
		}

		// once script has been read, evaluate it as normal
		return eval(script, context);
	}

	@Override
	public Object eval(String script, Bindings n) throws ScriptException {
		
		BSFManager bsfManager = new BSFManager();
		
		try {
			// requirements are to use passed Bindings n as engine scope but take the rest from default ScriptContext
			declareGlobalScopeBeansFromScriptContext(bsfManager, context);
			declareMiscScopeBeansFromScriptContext(bsfManager, context);
			declareBeansFromBindings(bsfManager, n);			
		
			// evaluate script
			return bsfManager.eval("rexx", "<unknown>", 0, 0, script);
		}
		catch (BSFException e) {
			throw new ScriptException(e);
		}
	}

	@Override
	public Object eval(Reader reader, Bindings n) throws ScriptException {

		String script = "";
		try {
			script = readStringFromReader(reader);
		}
		catch (IOException e) {
			throw new ScriptException(e);
		}

		// once script has been read, evaluate it as normal
		return eval(script, n);
	}

	@Override
	public Object get(String key) {
		return getBindings(ScriptContext.ENGINE_SCOPE).get(key);
	}

	@Override
	public Bindings getBindings(int scope) {
		// The Bindings instances that are returned must be identical to those returned by the getBindings() method of ScriptContext 
		// called with corresponding arguments on the default ScriptContext of the ScriptEngine.
		//
		// JSR Spec, quoted in http://today.java.net/article/2006/09/19/making-scripting-languages-jsr-223-aware
		
		return getContext().getBindings(scope);
	}

	@Override
	public ScriptContext getContext() {
		return context;
	}

	@Override
	public ScriptEngineFactory getFactory() {
		return factory;
	}

	@Override
	public void put(String key, Object value) {
		getBindings(ScriptContext.ENGINE_SCOPE).put(key, value);

	}

	@Override
	public void setBindings(Bindings bindings, int scope) {
		// The Bindings instances that are returned must be identical to those returned by the getBindings() method of ScriptContext 
		// called with corresponding arguments on the default ScriptContext of the ScriptEngine.
		//
		// JSR Spec, quoted in http://today.java.net/article/2006/09/19/making-scripting-languages-jsr-223-aware
		
		getContext().setBindings(bindings, scope);

	}

	@Override
	public void setContext(ScriptContext context) {
		this.context = context;

	}

}
